using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;

class NovaGodina
{
    static int n;
    static double epsilon;
    static double[] x, y;
    
    // MST edges as adjacency list
    static List<int>[] mstAdj;
    
    static double ManhattanDist(int i, int j)
    {
        return Math.Abs(x[i] - x[j]) + Math.Abs(y[i] - y[j]);
    }
    
    static void ComputeMST()
    {
        // Prim's algorithm - O(n^2)
        mstAdj = new List<int>[n];
        for (int i = 0; i < n; i++)
            mstAdj[i] = new List<int>();
        
        bool[] inMST = new bool[n];
        double[] minDist = new double[n];
        int[] parent = new int[n];
        
        for (int i = 0; i < n; i++)
        {
            minDist[i] = double.MaxValue;
            parent[i] = -1;
        }
        
        minDist[0] = 0;
        
        for (int iter = 0; iter < n; iter++)
        {
            // Find vertex with minimum distance not yet in MST
            int u = -1;
            double best = double.MaxValue;
            for (int i = 0; i < n; i++)
            {
                if (!inMST[i] && minDist[i] < best)
                {
                    best = minDist[i];
                    u = i;
                }
            }
            
            if (u == -1) break;
            inMST[u] = true;
            
            if (parent[u] != -1)
            {
                mstAdj[u].Add(parent[u]);
                mstAdj[parent[u]].Add(u);
            }
            
            // Update distances
            for (int v = 0; v < n; v++)
            {
                if (!inMST[v])
                {
                    double dist = ManhattanDist(u, v);
                    if (dist < minDist[v])
                    {
                        minDist[v] = dist;
                        parent[v] = u;
                    }
                }
            }
        }
    }
    
    static void Main()
    {
        // Read input
        string[] firstLine = Console.ReadLine().Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
        n = int.Parse(firstLine[0]);
        epsilon = double.Parse(firstLine[1], CultureInfo.InvariantCulture);
        
        x = new double[n];
        y = new double[n];
        
        for (int i = 0; i < n; i++)
        {
            string[] coords = Console.ReadLine().Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
            x[i] = double.Parse(coords[0], CultureInfo.InvariantCulture);
            y[i] = double.Parse(coords[1], CultureInfo.InvariantCulture);
        }
        
        // Compute MST
        ComputeMST();
        
        // Grid-based selection
        // Cell size should be epsilon/2 to ensure coverage
        double cellSize = epsilon / 2.0;
        
        // Map from grid cell to selected vertex
        Dictionary<(int, int), int> cellToVertex = new Dictionary<(int, int), int>();
        
        // For each vertex, determine its cell and select it if cell is empty
        for (int i = 0; i < n; i++)
        {
            int cx = (int)(x[i] / cellSize);
            int cy = (int)(y[i] / cellSize);
            var cell = (cx, cy);
            
            if (!cellToVertex.ContainsKey(cell))
            {
                cellToVertex[cell] = i;
            }
        }
        
        // Also need to cover edges - sample points along MST edges
        // and make sure cells containing edge points have representatives
        bool[] visited = new bool[n];
        Queue<int> queue = new Queue<int>();
        queue.Enqueue(0);
        visited[0] = true;
        
        HashSet<(int, int)> edgesProcessed = new HashSet<(int, int)>();
        
        while (queue.Count > 0)
        {
            int u = queue.Dequeue();
            foreach (int v in mstAdj[u])
            {
                var edgeKey = (Math.Min(u, v), Math.Max(u, v));
                if (edgesProcessed.Contains(edgeKey)) continue;
                edgesProcessed.Add(edgeKey);
                
                if (!visited[v])
                {
                    visited[v] = true;
                    queue.Enqueue(v);
                }
                
                // Sample points along edge (u, v)
                double edgeLen = ManhattanDist(u, v);
                int numSamples = Math.Max(2, (int)Math.Ceiling(edgeLen / (cellSize / 2)));
                
                for (int s = 0; s <= numSamples; s++)
                {
                    double t = (double)s / numSamples;
                    double px = x[u] + t * (x[v] - x[u]);
                    double py = y[u] + t * (y[v] - y[u]);
                    
                    int cx = (int)(px / cellSize);
                    int cy = (int)(py / cellSize);
                    var cell = (cx, cy);
                    
                    if (!cellToVertex.ContainsKey(cell))
                    {
                        // Find nearest vertex from P to this point
                        int best = -1;
                        double bestDist = double.MaxValue;
                        for (int i = 0; i < n; i++)
                        {
                            double dist = Math.Abs(x[i] - px) + Math.Abs(y[i] - py);
                            if (dist < bestDist)
                            {
                                bestDist = dist;
                                best = i;
                            }
                        }
                        cellToVertex[cell] = best;
                    }
                }
            }
        }
        
        // Collect selected vertices (Q)
        HashSet<int> selectedSet = new HashSet<int>(cellToVertex.Values);
        List<int> Q = selectedSet.ToList();
        Q.Sort();
        
        // Build spanning tree T over Q
        // We'll use MST structure to guide connectivity
        // First, map each original vertex to its nearest selected vertex
        int[] vertexToQ = new int[n];
        for (int i = 0; i < n; i++)
        {
            int best = -1;
            double bestDist = double.MaxValue;
            foreach (int q in Q)
            {
                double dist = ManhattanDist(i, q);
                if (dist < bestDist)
                {
                    bestDist = dist;
                    best = q;
                }
            }
            vertexToQ[i] = best;
        }
        
        // Build tree edges by following MST structure
        HashSet<(int, int)> treeEdges = new HashSet<(int, int)>();
        
        foreach (var edgeKey in edgesProcessed)
        {
            int u = edgeKey.Item1;
            int v = edgeKey.Item2;
            int qu = vertexToQ[u];
            int qv = vertexToQ[v];
            
            if (qu != qv)
            {
                var tEdge = (Math.Min(qu, qv), Math.Max(qu, qv));
                treeEdges.Add(tEdge);
            }
        }
        
        // The edges might not form a connected tree yet
        // Use BFS/DFS from MST to build a spanning tree over Q
        List<(int, int)> finalEdges = new List<(int, int)>();
        HashSet<int> inTree = new HashSet<int>();
        
        // Build adjacency for potential edges
        Dictionary<int, List<int>> potentialAdj = new Dictionary<int, List<int>>();
        foreach (int q in Q)
            potentialAdj[q] = new List<int>();
        
        foreach (var edge in treeEdges)
        {
            potentialAdj[edge.Item1].Add(edge.Item2);
            potentialAdj[edge.Item2].Add(edge.Item1);
        }
        
        // BFS to select tree edges
        if (Q.Count > 0)
        {
            Queue<int> bfs = new Queue<int>();
            bfs.Enqueue(Q[0]);
            inTree.Add(Q[0]);
            
            while (bfs.Count > 0)
            {
                int u = bfs.Dequeue();
                foreach (int v in potentialAdj[u])
                {
                    if (!inTree.Contains(v))
                    {
                        inTree.Add(v);
                        finalEdges.Add((u, v));
                        bfs.Enqueue(v);
                    }
                }
            }
        }
        
        // If not all vertices are connected, add more edges using MST path
        if (inTree.Count < Q.Count)
        {
            // Fallback: connect remaining vertices by finding paths in MST
            foreach (int q in Q)
            {
                if (!inTree.Contains(q))
                {
                    // BFS in original MST to find path to any vertex in tree
                    int[] parentPath = new int[n];
                    bool[] visitedPath = new bool[n];
                    for (int i = 0; i < n; i++) parentPath[i] = -1;
                    
                    Queue<int> pathBfs = new Queue<int>();
                    pathBfs.Enqueue(q);
                    visitedPath[q] = true;
                    
                    int found = -1;
                    while (pathBfs.Count > 0 && found == -1)
                    {
                        int u = pathBfs.Dequeue();
                        if (u != q && inTree.Contains(u))
                        {
                            found = u;
                            break;
                        }
                        foreach (int v in mstAdj[u])
                        {
                            if (!visitedPath[v])
                            {
                                visitedPath[v] = true;
                                parentPath[v] = u;
                                pathBfs.Enqueue(v);
                                
                                if (inTree.Contains(v))
                                {
                                    found = v;
                                    break;
                                }
                            }
                        }
                    }
                    
                    if (found != -1)
                    {
                        // Add edges along the path, using selected vertices
                        List<int> path = new List<int>();
                        int curr = q;
                        
                        // Reconstruct path from q towards found
                        // We need to walk from found back to q
                        List<int> pathReverse = new List<int>();
                        curr = found;
                        while (curr != -1)
                        {
                            pathReverse.Add(curr);
                            if (curr == q) break;
                            curr = parentPath[curr];
                        }
                        
                        // Actually we need the selected vertices along this path
                        int prev = -1;
                        foreach (int node in pathReverse)
                        {
                            int sq = vertexToQ[node];
                            if (selectedSet.Contains(sq) && !inTree.Contains(sq))
                            {
                                if (prev != -1)
                                {
                                    finalEdges.Add((prev, sq));
                                }
                                else if (inTree.Count > 0)
                                {
                                    // Connect to an existing tree node
                                    int connector = found;
                                    if (inTree.Contains(vertexToQ[found]))
                                        connector = vertexToQ[found];
                                    else
                                        connector = inTree.First();
                                    finalEdges.Add((connector, sq));
                                }
                                inTree.Add(sq);
                                prev = sq;
                            }
                            else if (inTree.Contains(sq))
                            {
                                if (prev != -1 && prev != sq)
                                {
                                    finalEdges.Add((prev, sq));
                                    prev = sq;
                                }
                                else
                                {
                                    prev = sq;
                                }
                            }
                        }
                    }
                }
            }
        }
        
        // Final fallback: if still not connected, just connect all Q vertices using MST on Q
        if (inTree.Count < Q.Count)
        {
            // Compute MST over just Q vertices
            int m = Q.Count;
            bool[] inMST2 = new bool[m];
            double[] minDist2 = new double[m];
            int[] parent2 = new int[m];
            
            for (int i = 0; i < m; i++)
            {
                minDist2[i] = double.MaxValue;
                parent2[i] = -1;
            }
            minDist2[0] = 0;
            
            finalEdges.Clear();
            
            for (int iter = 0; iter < m; iter++)
            {
                int u = -1;
                double best = double.MaxValue;
                for (int i = 0; i < m; i++)
                {
                    if (!inMST2[i] && minDist2[i] < best)
                    {
                        best = minDist2[i];
                        u = i;
                    }
                }
                
                if (u == -1) break;
                inMST2[u] = true;
                
                if (parent2[u] != -1)
                {
                    finalEdges.Add((Q[parent2[u]], Q[u]));
                }
                
                for (int v = 0; v < m; v++)
                {
                    if (!inMST2[v])
                    {
                        double dist = ManhattanDist(Q[u], Q[v]);
                        if (dist < minDist2[v])
                        {
                            minDist2[v] = dist;
                            parent2[v] = u;
                        }
                    }
                }
            }
        }
        
        // Output
        // Create index mapping from 0-based to 1-based
        Dictionary<int, int> qIndex = new Dictionary<int, int>();
        for (int i = 0; i < Q.Count; i++)
        {
            qIndex[Q[i]] = i + 1; // 1-based index in Q
        }
        
        Console.WriteLine(Q.Count);
        Console.WriteLine(string.Join(" ", Q.Select(q => q + 1))); // 1-based original indices
        
        foreach (var edge in finalEdges)
        {
            // Output as indices into Q (1-based)
            Console.WriteLine($"{qIndex[edge.Item1]} {qIndex[edge.Item2]}");
        }
    }
}